/*! \file 
**********************************************************************************
*Title:                         Discretix OMA DRM v2 Secure RO Handling ATP Test source file
*
* Filename:                     ODRM_SecROHandling_RoActivation_ATP.c 
*
* 
* Created:                      17.03.2009
*
*
* \Author                      Shelly Lerman
*
* \Remarks
*           Copyright (C) 2007 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/

/************* Include Files ***********************************/
#include "ODRM_SecROHandling_RoActivation_ATP.h"
#include "tlk_odrm_types.h"
#include "KMNG_Defs.h"
#include "tlk_sclk_api_types.h"
#include "ODRM_SecROHandling_data.h"
#include "DX_VOS_String.h"
#include "DX_VOS_Mem.h"
#include "ODRM_SecROHandling_Utils_ATP.h"
#include "MW_ATP_UTIL_funcs.h"
#include "KMNG_API.h"
#include "tlk_sclk_api.h"
#include "CRYS.h"


/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_ActivationTwoConstr
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test License Pre-Processing and Activation using two constrains RO (expired and valid license). 
* Algorithm:
* 1. Get workspace size. 
* 2. Create key ring and initialize it with device private key
* 3. Initialize Secure Clock descriptor
* 4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key .
* 5. Activate content license using  TLK_ODRM_RetrieveKey() with KMAC and KREK keys from step 4 and secure clock descriptor from step 3.
* 6. Validate that  this is correct RO key.
*    6.1 Encrypt Plain text with the activated RO key from TLK_ODRM_RetrieveKey().
*    6.2 compare it to the Encrypted plain text with the 
*        original RO key
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_ActivationTwoConstr(void)
{

    DxUint32_t TST_Error = 0;
    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};

    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDevice = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};
    TLK_ODRM_Buffer_t   TST_RightsXML = {0};
    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_AES_WrappedKey_t  TST_WrappedKey;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID ****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr        = (DxUint8_t*)TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);

    TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(TwoRestrRightsXmlStr);
    TST_RightsXML.buff_ptr = (DxByte*)TwoRestrRightsXmlStr;

    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n******************************************\n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"*** TST_ATP_SEC_RO_ActivationTwoConstr ***\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"******************************************\n \n"));

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
        "TST_ATP_SEC_RO_ActivationTwoConstr", 
        "TST_ATP_SEC_RO_ActivationTwoConstr");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
        "TST_ATP_SEC_RO_ActivationTwoConstr",
        "TST_ATP_SEC_RO_ActivationTwoConstr");


    /* 2. Create key ring and initialize it with device private key */
    TST_NumOfSymKeys = 0;
    TST_NumOfRSAKeys = 1;
    TST_NumOfDHKeys  = 0;
    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
        passwordBuff.buff_ptr,
        TST_NumOfSymKeys,
        TST_NumOfRSAKeys,
        TST_NumOfDHKeys,
        TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDevice",
        "TST_ATP_SEC_RO_ActivationTwoConstr",
        "TST_ATP_SEC_RO_ActivationTwoConstr");

    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
        passwordBuff.buff_ptr,
        TST_NumOfSymKeys,
        TST_NumOfRSAKeys,
        TST_NumOfDHKeys,
        TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
        "TST_ATP_SEC_RO_ActivationTwoConstr",
        "TST_ATP_SEC_RO_ActivationTwoConstr");

    TST_KeyDevice.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDevice.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;


    TST_Error = KMNG_ImportRsaUserKey(passwordBuff.buff_ptr, 
        passwordBuff.buffSizeInBytes,
        passwordBuff.buff_ptr, 
        passwordBuff.buffSizeInBytes,
        KMNG_KeyTypeRSAPair,                    /*keyType*/
        ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES, /*keySize*/ 
        KMNG_KEY_USAGE_ODRM | KMNG_KEY_USAGE_SIGNING,                    /*keyUsage*/
        TLK_ODRM_KEY_RESTRICTION,               /*keyRestriction*/
        TST_UserSpecificKeyData,                    /*UserSpecificKeyData*/
        PrivModulus,                           /*modulus_ptr*/
        MODULUS_SIZE,                       /*modulusSize*/
        PubExponent,                       /*E_ptr*/
        PUB_EXP_SIZE,                    /*E_Size*/
        PrivExponent,                       /*D_ptr*/
        PRIV_EXP_SIZE,                   /*D_Size*/
        &TST_KeyDevice.keyId,
        TST_gKeyRingDevice);


    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportRsaUserKey ",
        "TST_ATP_SEC_RO_ActivationTwoConstr",
        "TST_ATP_SEC_RO_ActivationTwoConstr");

    keyRingDeviceStruct.keyKmngRing_ptr = TST_gKeyRingDevice;
    keyRingDeviceStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDeviceStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /*3.Or initialize secure clock descriptor*/
    TST_Error = TLK_SCLK_Init(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_SCLK_Init",
        "TST_ATP_SEC_RO_ActivationTwoConstr",
        "TST_ATP_SEC_RO_ActivationTwoConstr");

    TST_Error = TST_SCLK_Init(&TST_ServiceDescr);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TST_SCLK_Init",
        "TST_ATP_SEC_RO_ActivationTwoConstr",
        "TST_ATP_SEC_RO_ActivationTwoConstr");

    /*4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key */
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_RightsXML,
        &keyRingDeviceStruct,
        &TST_KeyDevice,
        &keyRingKmacKrecStruct,
        &TST_kmacKey,
        &TST_krecKey,
        RO_TST_gWorkspace,
        RO_TST_gWorkspaceSizeInBytes);


    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall",
        "TST_ATP_SEC_RO_ActivationTwoConstr",
        "TST_ATP_SEC_RO_ActivationTwoConstr");


    /*5. Activate content license using  TLK_ODRM_RetrieveKey() with KMAC 
    and KREK keys from step 4 and secure clock descriptor from step 3.*/
    TST_Error = TLK_ODRM_RetrieveKey(&TST_RightsXML,
                                     &keyRingKmacKrecStruct,
                                     &TST_kmacKey,
                                     &TST_krecKey,
                                     "cid:cid6108_jpg",
                                     &TST_ServiceDescr,
                                     DX_NULL,
                                     DX_NULL,
                                     DX_NULL,
                                     DX_NULL,
                                     0,
                                     TST_WrappedKey,
                                     RO_TST_gWorkspace,
                                     RO_TST_gWorkspaceSizeInBytes);

    /*6. Verify that TLK_ODRM_RC_ERROR_NO_VALID_RIGTHS_FOUND_IN_RO error was returned */
    if(TST_Error != TLK_ODRM_RC_ERROR_NO_VALID_RIGTHS_FOUND_IN_RO)
        TST_Error = TST_FAIL;
    else
        TST_Error = DX_SUCCESS;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_RetrieveKey with expired and valid constrains. Test ",
        "TST_ATP_SEC_RO_ActivationTwoConstr",
        "TST_ATP_SEC_RO_ActivationTwoConstr");

EXIT_ON_ERROR:
    TLK_SCLK_Terminate(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    return TST_Error;
}




/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_ExpiredROActivation
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test License Activation using expired license
* Algorithm:
*1. Get workspace size. 
*2. Create key ring and initialize it with device private key
*3. Call to  TLK_ODRM_ProcessRegistration() API for RI Issuer registration
*4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key .
*5. Activate content license using  TLK_ODRM_RetrieveKey() with KMAC and
*   KREK keys from step 4, secure clock descriptor from step 3 and RO XML with    expired rights.
*6. Verify that TLK_ODRM_RC_ERROR_NO_VALID_RIGTHS_FOUND_IN_RO error was returned
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_ExpiredROActivation(void)
{

    DxUint32_t TST_Error = 0;

    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};

    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDevice = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};
    TLK_ODRM_Buffer_t   TST_RightsXML = {0};
    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_AES_WrappedKey_t  TST_WrappedKey;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID ****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr        = TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);
    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n******************************************** \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"***** TST_ATP_SEC_RO_ExpiredROActivation ***\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"********************************************\n \n"));


    /*Initialize rights  struct*/
    TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(ExpiredRightsXmlStr);
    TST_RightsXML.buff_ptr = (DxByte*)ExpiredRightsXmlStr;

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
                               "TST_ATP_SEC_RO_ExpiredROActivation", 
                               "TST_ATP_SEC_RO_ExpiredROActivation");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
                               "TST_ATP_SEC_RO_ExpiredROActivation",
                               "TST_ATP_SEC_RO_ExpiredROActivation");


    /* 2. Create key ring and initialize it with device private key */
    TST_NumOfSymKeys = 0;
    TST_NumOfRSAKeys = 1;
    TST_NumOfDHKeys  = 0;
    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                  passwordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDevice",
                               "TST_ATP_SEC_RO_ExpiredROActivation",
                               "TST_ATP_SEC_RO_ExpiredROActivation");

    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                 passwordBuff.buff_ptr,
                                 TST_NumOfSymKeys,
                                 TST_NumOfRSAKeys,
                                 TST_NumOfDHKeys,
                                 TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_ExpiredROActivation",
                               "TST_ATP_SEC_RO_ExpiredROActivation");

    TST_KeyDevice.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDevice.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_Error = KMNG_ImportRsaUserKey(passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      KMNG_KeyTypeRSAPair,                    /*keyType*/
                                      ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES, /*keySize*/ 
                                      KMNG_KEY_USAGE_ODRM | KMNG_KEY_USAGE_SIGNING,                    /*keyUsage*/
                                      TLK_ODRM_KEY_RESTRICTION,               /*keyRestriction*/
                                      TST_UserSpecificKeyData,                    /*UserSpecificKeyData*/
                                      PrivModulus,                           /*modulus_ptr*/
                                      MODULUS_SIZE,                       /*modulusSize*/
                                      PubExponent,                       /*E_ptr*/
                                      PUB_EXP_SIZE,                    /*E_Size*/
                                      PrivExponent,                       /*D_ptr*/
                                      PRIV_EXP_SIZE,                   /*D_Size*/
                                      &TST_KeyDevice.keyId,
                                      TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportRsaUserKey ",
                               "TST_ATP_SEC_RO_ExpiredROActivation",
                               "TST_ATP_SEC_RO_ExpiredROActivation");

    keyRingDeviceStruct.keyKmngRing_ptr = TST_gKeyRingDevice;
    keyRingDeviceStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDeviceStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /*3.Or initialize secure cklock descriptor*/
    TST_Error = TLK_SCLK_Init(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_SCLK_Init",
                               "TST_ATP_SEC_RO_ExpiredROActivation",
                               "TST_ATP_SEC_RO_ExpiredROActivation");

    TST_Error = TST_SCLK_Init(&TST_ServiceDescr);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TST_SCLK_Init",
                               "TST_ATP_SEC_RO_ExpiredROActivation",
                               "TST_ATP_SEC_RO_ExpiredROActivation");


    /*4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key . */
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_RightsXML,
                                                     &keyRingDeviceStruct,
                                                     &TST_KeyDevice,
                                                     &keyRingKmacKrecStruct,
                                                     &TST_kmacKey,
                                                     &TST_krecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall",
                               "TST_ATP_SEC_RO_ExpiredROActivation",
                               "TST_ATP_SEC_RO_ExpiredROActivation");

    /*5. Activate content license using  TLK_ODRM_RetrieveKey() with KMAC and
    KREK keys from step 4, secure clock descriptor from step 3 and RO XML with 
    expired rights.*/
    TST_Error = TLK_ODRM_RetrieveKey(&TST_RightsXML,
                                     &keyRingKmacKrecStruct,
                                     &TST_kmacKey,
                                     &TST_krecKey,
                                     "cid:cid6093_mp3",
                                     &TST_ServiceDescr,
                                     DX_NULL,
                                     DX_NULL,
                                     DX_NULL,
                                     DX_NULL,
                                     0,
                                     TST_WrappedKey,
                                     RO_TST_gWorkspace,
                                     RO_TST_gWorkspaceSizeInBytes);

    /*6. Verify that TLK_ODRM_RC_ERROR_NO_VALID_RIGTHS_FOUND_IN_RO error was returned */
    if(TST_Error != TLK_ODRM_RC_ERROR_NO_VALID_RIGTHS_FOUND_IN_RO)
        TST_Error = TST_FAIL;
    else
        TST_Error = DX_SUCCESS;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_RetrieveKey succeed with expired RO . Test ",
                               "TST_ATP_SEC_RO_ExpiredROActivation",
                               "TST_ATP_SEC_RO_ExpiredROActivation");

EXIT_ON_ERROR:
    TLK_SCLK_Terminate(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    return TST_Error;
}

/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_GroupROActivation
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test activation of license with group ID in Secure RO Handling 
* Algorithm:
* 1. Get workspace size. 
* 2. Create key ring and initialize it with device private key
* 3. Call to  TLK_ODRM_ProcessRegistration() API  for RI Issuer registration
* 4. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key and RO XML with group ID.
* 5. Activate content license using  TLK_ODRM_RetrieveKey() with keys from step 4 and secure clock descriptor from step 3 and encrypted group key.
* 6. Validate that  this is correct RO key.
*    6.1  Encrypt Plain text with the activated RO key from    TLK_ODRM_RetrieveKey().
*    6.2 compare it to the Encrypted plain text with the 
*        original RO key
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_GroupROActivation(void)
{

    DxUint32_t TST_Error = 0;
    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};

    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDevice = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};
    TLK_ODRM_Buffer_t   TST_GroupKey = {0};
    TLK_ODRM_Buffer_t   TST_RightsXML = {0};
    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_AES_WrappedKey_t  TST_WrappedKey;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID ****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr        = TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);
    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n******************************************** \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"***** TST_ATP_SEC_RO_GroupROActivation *****\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"********************************************\n \n"));

    /*Initialize RO XML  str*/
    TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(GIDRightsXmlStr);
    TST_RightsXML.buff_ptr = (DxByte*)GIDRightsXmlStr;

    /*Initialize group key buffer*/
    TST_GroupKey.buffSizeInBytes = 48;
    TST_GroupKey.buff_ptr = (DxByte*)TST_gGroupIDKey;

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
                               "TST_ATP_SEC_RO_GroupROActivation", 
                               "TST_ATP_SEC_RO_GroupROActivation");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
                               "TST_ATP_SEC_RO_GroupROActivation",
                               "TST_ATP_SEC_RO_GroupROActivation");


    /* 2. Create key ring and initialize it with device private key */
    TST_NumOfSymKeys = 0;
    TST_NumOfRSAKeys = 1;
    TST_NumOfDHKeys  = 0;
    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                  passwordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDevice",
                               "TST_ATP_SEC_RO_GroupROActivation",
                               "TST_ATP_SEC_RO_GroupROActivation");

    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                 passwordBuff.buff_ptr,
                                 TST_NumOfSymKeys,
                                 TST_NumOfRSAKeys,
                                 TST_NumOfDHKeys,
                                 TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_GroupROActivation",
                               "TST_ATP_SEC_RO_GroupROActivation");

    TST_KeyDevice.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDevice.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;


    TST_Error = KMNG_ImportRsaUserKey(passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      KMNG_KeyTypeRSAPair,                    /*keyType*/
                                      ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES, /*keySize*/ 
                                      KMNG_KEY_USAGE_ODRM | KMNG_KEY_USAGE_SIGNING,                    /*keyUsage*/
                                      TLK_ODRM_KEY_RESTRICTION,               /*keyRestriction*/
                                      TST_UserSpecificKeyData,                    /*UserSpecificKeyData*/
                                      PrivModulus,                           /*modulus_ptr*/
                                      MODULUS_SIZE,                       /*modulusSize*/
                                      PubExponent,                       /*E_ptr*/
                                      PUB_EXP_SIZE,                    /*E_Size*/
                                      PrivExponent,                       /*D_ptr*/
                                      PRIV_EXP_SIZE,                   /*D_Size*/
                                      &TST_KeyDevice.keyId,
                                      TST_gKeyRingDevice);


    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportRsaUserKey ",
                               "TST_ATP_SEC_RO_GroupROActivation",
                               "TST_ATP_SEC_RO_GroupROActivation");

    keyRingDeviceStruct.keyKmngRing_ptr = TST_gKeyRingDevice;
    keyRingDeviceStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDeviceStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /*3. Initialize secure cklock descriptor */
    TST_Error = TLK_SCLK_Init(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_SCLK_Init",
                               "TST_ATP_SEC_RO_GroupROActivation",
                               "TST_ATP_SEC_RO_GroupROActivation");

    TST_Error = TST_SCLK_Init(&TST_ServiceDescr);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TST_SCLK_Init",
                               "TST_ATP_SEC_RO_GroupROActivation",
                               "TST_ATP_SEC_RO_GroupROActivation");


    /*4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key */
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_RightsXML,
                                                     &keyRingDeviceStruct,
                                                     &TST_KeyDevice,
                                                     &keyRingKmacKrecStruct,
                                                     &TST_kmacKey,
                                                     &TST_krecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall",
                               "TST_ATP_SEC_RO_GroupROActivation",
                               "TST_ATP_SEC_RO_GroupROActivation");

    /*6. Activate content license using  TLK_ODRM_RetrieveKey() with KMAC 
    and KREK keys from step 4 and secure clock descriptor from step 3.*/
    TST_Error = TLK_ODRM_RetrieveKey(&TST_RightsXML,
                                     &keyRingKmacKrecStruct,
                                     &TST_kmacKey,
                                     &TST_krecKey,
                                     "gid6051",
                                     &TST_ServiceDescr,
                                     DX_NULL,
                                     DX_NULL,
                                     DX_NULL,
                                     &TST_GroupKey,
                                     DX_ENCR_AES_128_CBC,
                                     TST_WrappedKey,
                                     RO_TST_gWorkspace,
                                     RO_TST_gWorkspaceSizeInBytes);


    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_RetrieveKey",
                               "TST_ATP_SEC_RO_GroupROActivation",
                               "TST_ATP_SEC_RO_GroupROActivation");

    /* 7.Validate that  this is correct RO key.*/

    /*7.1  Encrypt Plain text with the activated RO key from    TLK_ODRM_RetrieveKey().*/     
    TST_Error = CRYS_KMNG_AES(TST_WrappedKey,
                              IvCounter,
                              CRYS_AES_Encrypt,
                              CRYS_AES_ECB_mode,
                              TST_PlainBuffCid6051,
                              ODRMTLK_TST_AES_KEY_SIZE,
                              TST_DataoutBuff);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "CRYS_KMNG_AES",
                               "TST_ATP_SEC_RO_GroupROActivation",
                               "TST_ATP_SEC_RO_GroupROActivation");

    /* 7.2 compare it to the Encrypted plain text with the original RO key.*/
    TST_Error = DX_VOS_MemCmp(TST_DataoutBuff, TST_EncryptBuffCid6051, TEMP_BUFF_LEN);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "DX_VOS_MemCmp",
                               "TST_ATP_SEC_RO_GroupROActivation",
                               "TST_ATP_SEC_RO_GroupROActivation");
EXIT_ON_ERROR:
    TLK_SCLK_Terminate(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    return TST_Error;
}


/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_ROActivationWithInvGroupKey
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test activation of license with invalid group ID in Secure RO Handling 
* Algorithm:
* 1. Get workspace size. 
* 2. Create key ring and initialize it with device private key
* 3. Call to  TLK_ODRM_ProcessRegistration() API  for RI Issuer registration
* 4. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key and RO XML with group ID.
* 5. Activate content license using  TLK_ODRM_RetrieveKey() with keys from step 4 and secure clock descriptor from step 3 and without encrypted group key.
* 6. Verify that incorrect key was returned
* 7. Activate content license using  TLK_ODRM_RetrieveKey() with keys from step 4 and secure clock descriptor from step 3 and invalid encrypted group key.
* 8. Verify that incorrect key error was returned
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_ROActivationWithInvGroupKey(void)
{

    DxUint32_t TST_Error = 0;
    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};

    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDevice = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};
    TLK_ODRM_Buffer_t   TST_GroupKey = {0};
    TLK_ODRM_Buffer_t   TST_RightsXML = {0};
    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_AES_WrappedKey_t  TST_WrappedKey;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID ****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr        = TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);
    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n**************************************************** \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"*****TST_ATP_SEC_RO_ROActivationWithInvGroupKey*****\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"****************************************************\n \n"));

    /*Initialize RO XML  str*/
    TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(GIDRightsXmlStr);
    TST_RightsXML.buff_ptr = (DxByte*)GIDRightsXmlStr;

   
    /*Initialize group key buffer*/
    TST_GroupKey.buffSizeInBytes = 48;
    TST_GroupKey.buff_ptr = (DxByte*)TST_gInvalidGroupIDKey;

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
        "TST_ATP_SEC_RO_ROActivationWithInvGroupKey", 
        "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
        "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
        "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");


    /* 2. Create key ring and initialize it with device private key */
    TST_NumOfSymKeys = 0;
    TST_NumOfRSAKeys = 1;
    TST_NumOfDHKeys  = 0;
    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                  passwordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDevice",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                 passwordBuff.buff_ptr,
                                 TST_NumOfSymKeys,
                                 TST_NumOfRSAKeys,
                                 TST_NumOfDHKeys,
                                 TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

    TST_KeyDevice.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDevice.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;


    TST_Error = KMNG_ImportRsaUserKey(passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      KMNG_KeyTypeRSAPair,                    /*keyType*/
                                      ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES, /*keySize*/ 
                                      KMNG_KEY_USAGE_ODRM | KMNG_KEY_USAGE_SIGNING,                    /*keyUsage*/
                                      TLK_ODRM_KEY_RESTRICTION,               /*keyRestriction*/
                                      TST_UserSpecificKeyData,                    /*UserSpecificKeyData*/
                                      PrivModulus,                           /*modulus_ptr*/
                                      MODULUS_SIZE,                       /*modulusSize*/
                                      PubExponent,                       /*E_ptr*/
                                      PUB_EXP_SIZE,                    /*E_Size*/
                                      PrivExponent,                       /*D_ptr*/
                                      PRIV_EXP_SIZE,                   /*D_Size*/
                                      &TST_KeyDevice.keyId,
                                      TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportRsaUserKey ",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

    keyRingDeviceStruct.keyKmngRing_ptr = TST_gKeyRingDevice;
    keyRingDeviceStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDeviceStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /*3. Initialize secure cklock descriptor */

    TST_Error = TLK_SCLK_Init(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_SCLK_Init",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

    TST_Error = TST_SCLK_Init(&TST_ServiceDescr);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TST_SCLK_Init",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

    /*4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key */
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_RightsXML,
                                                     &keyRingDeviceStruct,
                                                     &TST_KeyDevice,
                                                     &keyRingKmacKrecStruct,
                                                     &TST_kmacKey,
                                                     &TST_krecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall",
        "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
        "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

    /*5. Activate content license using  TLK_ODRM_RetrieveKey() with keys from step 4
    and secure clock descriptor from step 3 and without encrypted group key.*/
    TST_Error = TLK_ODRM_RetrieveKey(&TST_RightsXML,
                                     &keyRingKmacKrecStruct,
                                     &TST_kmacKey,
                                     &TST_krecKey,
                                     "gid6051",
                                     &TST_ServiceDescr,
                                     DX_NULL,
                                     DX_NULL,
                                     DX_NULL,
                                     DX_NULL,
                                     0,
                                     TST_WrappedKey,
                                     RO_TST_gWorkspace,
                                     RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_RetrieveKey of License with GI without GID key succeed. Test  ",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

    /* 6.Validate that incorrect RO key was returned.*/
    /*6.1  Encrypt Plain text with the activated RO key from    TLK_ODRM_RetrieveKey().*/     
    TST_Error = CRYS_KMNG_AES(TST_WrappedKey,
                              IvCounter,
                              CRYS_AES_Encrypt,
                              CRYS_AES_ECB_mode,
                              TST_PlainBuffCid6051,
                              ODRMTLK_TST_AES_KEY_SIZE,
                              TST_DataoutBuff);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "CRYS_KMNG_AES",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

    /* 6.2 compare it to the Encrypted plain text with the original RO key.*/
    TST_Error = DX_VOS_MemCmp(TST_DataoutBuff, TST_EncryptBuffCid6051, TEMP_BUFF_LEN);

    if(TST_Error == DX_SUCCESS)
        TST_Error = TST_FAIL;
    else
        TST_Error = DX_SUCCESS;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "DX_VOS_MemCmp",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

    /*7. Activate content license using  TLK_ODRM_RetrieveKey() with keys from step 4
    and secure clock descriptor from step 3 and invalid encrypted group key.*/
    TST_Error = TLK_ODRM_RetrieveKey(&TST_RightsXML,
                                     &keyRingKmacKrecStruct,
                                     &TST_kmacKey,
                                     &TST_krecKey,
                                     "gid6051",
                                     &TST_ServiceDescr,
                                     DX_NULL,
                                     DX_NULL,
                                     DX_NULL,
                                     &TST_GroupKey,
                                     DX_ENCR_AES_128_CBC,
                                     TST_WrappedKey,
                                     RO_TST_gWorkspace,
                                     RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_RetrieveKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

    /* 8.Validate that incorrect RO key was returned.*/

    /*8.1  Encrypt Plain text with the activated RO key from    TLK_ODRM_RetrieveKey().*/     
    TST_Error = CRYS_KMNG_AES(TST_WrappedKey,
                              IvCounter,
                              CRYS_AES_Encrypt,
                              CRYS_AES_ECB_mode,
                              TST_PlainBuffCid6051,
                              ODRMTLK_TST_AES_KEY_SIZE,
                              TST_DataoutBuff);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "CRYS_KMNG_AES",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

    /* 8.2 compare it to the Encrypted plain text with the original RO key.*/
    TST_Error = DX_VOS_MemCmp(TST_DataoutBuff, TST_EncryptBuffCid6051, TEMP_BUFF_LEN);

    if(TST_Error == DX_SUCCESS)
        TST_Error = TST_FAIL;
    else
        TST_Error = DX_SUCCESS;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "DX_VOS_MemCmp",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey",
                               "TST_ATP_SEC_RO_ROActivationWithInvGroupKey");

EXIT_ON_ERROR:
    TLK_SCLK_Terminate(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    return TST_Error;
}


/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_ParentChiledROActivation
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test activation of license with parent license object in Secure RO Handling 
* Algorithm:
* 1. Get workspace size. 
*2. Create key ring and initialize it with device private key
*3. Call to  TLK_ODRM_ProcessRegistration() API  for RI Issuer registration
*4. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key and RO XML.
*5. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key and Parent RO XML.
*6. Activate content license using  TLK_ODRM_RetrieveKey() with keys from step 4, 5 and secure clock descriptor from step 3.
*7. Validate that  this is correct RO key.
*   7.1 Encrypt Plain text with the activated RO key from    TLK_ODRM_RetrieveKey().
*   7.2 compare it to the Encrypted plain text with the 
*       original RO key
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_ParentChiledROActivation(void)
{

    DxUint32_t TST_Error = 0;
    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};

    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};
    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecParentStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDevice = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};

    TLK_ODRM_KMNGKey_t  TST_ParentKmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_ParentKrecKey = {0};

    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_AES_WrappedKey_t  TST_WrappedKey;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    TLK_ODRM_Buffer_t   TST_ParentRightsXML = {0};

    TLK_ODRM_Buffer_t   TST_ChildRightsXML = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID ****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr        = TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);
    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n*********************************************** \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"*** TST_ATP_SEC_RO_ParentChiledROActivation ***\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"***********************************************\n \n"));


    /*Initialize parent RO XML structure*/
    TST_ParentRightsXML.buffSizeInBytes = DX_VOS_StrLen(ParentRightsXmlStr);
    TST_ParentRightsXML.buff_ptr = (DxByte*)ParentRightsXmlStr;


    /*Initialize child RO XML structure*/
    TST_ChildRightsXML.buffSizeInBytes = DX_VOS_StrLen(ChildRightsXmlStr);
    TST_ChildRightsXML.buff_ptr = (DxByte*)ChildRightsXmlStr;

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
        "TST_ATP_SEC_RO_ParentChiledROActivation", 
        "TST_ATP_SEC_RO_ParentChiledROActivation");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");


    /* 2. Create key ring and initialize it with device private key */
    TST_NumOfSymKeys = 0;
    TST_NumOfRSAKeys = 1;
    TST_NumOfDHKeys  = 0;
    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
        passwordBuff.buff_ptr,
        TST_NumOfSymKeys,
        TST_NumOfRSAKeys,
        TST_NumOfDHKeys,
        TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDevice",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");

    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
        passwordBuff.buff_ptr,
        TST_NumOfSymKeys,
        TST_NumOfRSAKeys,
        TST_NumOfDHKeys,
        TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");
    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
        passwordBuff.buff_ptr,
        TST_NumOfSymKeys,
        TST_NumOfRSAKeys,
        TST_NumOfDHKeys,
        TST_gParentKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gParentKeyRingKmacKrec",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");

    TST_KeyDevice.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDevice.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_ParentKmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_ParentKmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_ParentKrecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_ParentKrecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;


    TST_Error = KMNG_ImportRsaUserKey(passwordBuff.buff_ptr, 
        passwordBuff.buffSizeInBytes,
        passwordBuff.buff_ptr, 
        passwordBuff.buffSizeInBytes,
        KMNG_KeyTypeRSAPair,                    /*keyType*/
        ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES, /*keySize*/ 
        KMNG_KEY_USAGE_ODRM | KMNG_KEY_USAGE_SIGNING,                    /*keyUsage*/
        TLK_ODRM_KEY_RESTRICTION,               /*keyRestriction*/
        TST_UserSpecificKeyData,                    /*UserSpecificKeyData*/
        PrivModulus,                           /*modulus_ptr*/
        MODULUS_SIZE,                       /*modulusSize*/
        PubExponent,                       /*E_ptr*/
        PUB_EXP_SIZE,                    /*E_Size*/
        PrivExponent,                       /*D_ptr*/
        PRIV_EXP_SIZE,                   /*D_Size*/
        &TST_KeyDevice.keyId,
        TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportRsaUserKey ",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");

    keyRingDeviceStruct.keyKmngRing_ptr = TST_gKeyRingDevice;
    keyRingDeviceStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDeviceStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecParentStruct.keyKmngRing_ptr = TST_gParentKeyRingKmacKrec;
    keyRingKmacKrecParentStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecParentStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;


    /*3. Initialize secure cklock descriptor */

    TST_Error = TLK_SCLK_Init(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_SCLK_Init",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");

    TST_Error = TST_SCLK_Init(&TST_ServiceDescr);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TST_SCLK_Init",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");

    /*4. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key and child RO XML*/
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_ChildRightsXML,
        &keyRingDeviceStruct,
        &TST_KeyDevice,
        &keyRingKmacKrecStruct,
        &TST_kmacKey,
        &TST_krecKey,
        RO_TST_gWorkspace,
        RO_TST_gWorkspaceSizeInBytes);


    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall- Child RO",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");

    /*5. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key and Parent RO XML*/
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_ParentRightsXML,
        &keyRingDeviceStruct,
        &TST_KeyDevice,
        &keyRingKmacKrecParentStruct,
        &TST_ParentKmacKey,
        &TST_ParentKrecKey,
        RO_TST_gWorkspace,
        RO_TST_gWorkspaceSizeInBytes);


    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall- Parent RO",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");

    /*6. Activate content license using  TLK_ODRM_RetrieveKey() with keys from step 4, 5 
    and secure clock descriptor from step 3*/
    TST_Error = TLK_ODRM_RetrieveKey(&TST_ChildRightsXML,
        &keyRingKmacKrecStruct,
        &TST_kmacKey,
        &TST_krecKey,
        "cid:cid6121-2_jpg",
        &TST_ServiceDescr,
        &TST_ParentRightsXML,
        &keyRingKmacKrecParentStruct,
        &TST_ParentKmacKey,
        DX_NULL,
        0,
        TST_WrappedKey,
        RO_TST_gWorkspace,
        RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_RetrieveKey",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");

    /* 7.Validate that  this is correct RO key.*/

    /*7.1  Encrypt Plain text with the activated RO key from    TLK_ODRM_RetrieveKey().*/     
    TST_Error = CRYS_KMNG_AES(TST_WrappedKey,
        IvCounter,
        CRYS_AES_Encrypt,
        CRYS_AES_ECB_mode,
        TST_PlainBuffCid61211,
        ODRMTLK_TST_AES_KEY_SIZE,
        TST_DataoutBuff);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "CRYS_KMNG_AES",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");

    /* 7.2 compare it to the Encrypted plain text with the original RO key.*/
    TST_Error = DX_VOS_MemCmp(TST_DataoutBuff, TST_EncryptBuffCid61211, TEMP_BUFF_LEN);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "DX_VOS_MemCmp",
        "TST_ATP_SEC_RO_ParentChiledROActivation",
        "TST_ATP_SEC_RO_ParentChiledROActivation");

EXIT_ON_ERROR:
    TLK_SCLK_Terminate(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    return TST_Error;
}




/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_InvalidParentROActivation
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test activation of license with invalid Parent RO XML Secure RO Handling
* Algorithm:
* 1. Get workspace size. 
* 2. Create key ring and initialize it with device private key
* 3. Call to  TLK_ODRM_ProcessRegistration() API  for RI Issuer registration
* 4. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key and RO XML.
* 5. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key and Parent RO XML.
* 6. Activate content license using  TLK_ODRM_RetrieveKey() with keys from step 4, 5, secure clock descriptor from step 3and invalid Parent RO XML (RO integrity).
* 7. Activate content license using  TLK_ODRM_RetrieveKey() with keys from step 4, 5, secure clock descriptor from step 3 and invalid key in RO XML. 
* 8. Verify that TLK_ODRM_RC_ERROR_RO_MAC_VERIFICATION_FAILED error was returned
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_InvalidParentROActivation(void)
{

    DxUint32_t TST_Error = 0;
    DxUint8 index = 0;
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};
    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecParentStruct = {0};
    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDevice = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};

    TLK_ODRM_KMNGKey_t  TST_ParentKmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_ParentKrecKey = {0};

    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_AES_WrappedKey_t  TST_WrappedKey;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    TLK_ODRM_Buffer_t   TST_ParentRightsXML = {0};

    TLK_ODRM_Buffer_t   TST_ChildRightsXML = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID ****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr        = TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);
    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n**************************************************** \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"***** TST_ATP_SEC_RO_InvalidParentROActivation *****\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"****************************************************\n \n"));


    /*Initialize parent RO XML structure*/
    TST_ParentRightsXML.buffSizeInBytes = DX_VOS_StrLen(ParentRightsXmlStr);
    TST_ParentRightsXML.buff_ptr = (DxByte*)ParentRightsXmlStr;

    /*Initialize child RO XML structure*/
    TST_ChildRightsXML.buffSizeInBytes = DX_VOS_StrLen(ChildRightsXmlStr);
    TST_ChildRightsXML.buff_ptr = (DxByte*)ChildRightsXmlStr;

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
                              "TST_ATP_SEC_RO_InvalidParentROActivation", 
                              "TST_ATP_SEC_RO_InvalidParentROActivation");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
                               "TST_ATP_SEC_RO_InvalidParentROActivation",
                               "TST_ATP_SEC_RO_InvalidParentROActivation");


    /* 2. Create key ring and initialize it with device private key */
    TST_NumOfSymKeys = 0;
    TST_NumOfRSAKeys = 1;
    TST_NumOfDHKeys  = 0;
    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                  passwordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDevice",
                               "TST_ATP_SEC_RO_InvalidParentROActivation",
                               "TST_ATP_SEC_RO_InvalidParentROActivation");

    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                 passwordBuff.buff_ptr,
                                 TST_NumOfSymKeys,
                                 TST_NumOfRSAKeys,
                                 TST_NumOfDHKeys,
                                 TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_InvalidParentROActivation",
                               "TST_ATP_SEC_RO_InvalidParentROActivation");

    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;
    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                 passwordBuff.buff_ptr,
                                 TST_NumOfSymKeys,
                                 TST_NumOfRSAKeys,
                                 TST_NumOfDHKeys,
                                 TST_gParentKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gParentKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_InvalidParentROActivation",
                               "TST_ATP_SEC_RO_InvalidParentROActivation");

    TST_KeyDevice.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDevice.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_ParentKmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_ParentKmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_ParentKrecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_ParentKrecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;


    TST_Error = KMNG_ImportRsaUserKey(passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      KMNG_KeyTypeRSAPair,                    /*keyType*/
                                      ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES, /*keySize*/ 
                                      KMNG_KEY_USAGE_ODRM | KMNG_KEY_USAGE_SIGNING,                    /*keyUsage*/
                                      TLK_ODRM_KEY_RESTRICTION,               /*keyRestriction*/
                                      TST_UserSpecificKeyData,                    /*UserSpecificKeyData*/
                                      PrivModulus,                           /*modulus_ptr*/
                                      MODULUS_SIZE,                       /*modulusSize*/
                                      PubExponent,                       /*E_ptr*/
                                      PUB_EXP_SIZE,                    /*E_Size*/
                                      PrivExponent,                       /*D_ptr*/
                                      PRIV_EXP_SIZE,                   /*D_Size*/
                                      &TST_KeyDevice.keyId,
                                      TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportRsaUserKey ",
                               "TST_ATP_SEC_RO_InvalidParentROActivation",
                               "TST_ATP_SEC_RO_InvalidParentROActivation");

    keyRingDeviceStruct.keyKmngRing_ptr = TST_gKeyRingDevice;
    keyRingDeviceStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDeviceStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecParentStruct.keyKmngRing_ptr = TST_gParentKeyRingKmacKrec;
    keyRingKmacKrecParentStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecParentStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /*3. Initialize secure cklock descriptor */

    TST_Error = TLK_SCLK_Init(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_SCLK_Init",
                               "TST_ATP_SEC_RO_InvalidParentROActivation",
                               "TST_ATP_SEC_RO_InvalidParentROActivation");

    TST_Error = TST_SCLK_Init(&TST_ServiceDescr);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TST_SCLK_Init",
                               "TST_ATP_SEC_RO_InvalidParentROActivation",
                               "TST_ATP_SEC_RO_InvalidParentROActivation");


    /*4. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key and child RO XML*/
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_ChildRightsXML,
                                                     &keyRingDeviceStruct,
                                                     &TST_KeyDevice,
                                                     &keyRingKmacKrecStruct,
                                                     &TST_kmacKey,
                                                     &TST_krecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);


    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall- Child RO",
                               "TST_ATP_SEC_RO_InvalidParentROActivation",
                               "TST_ATP_SEC_RO_InvalidParentROActivation");

    /*5. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key and Parent RO XML*/
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_ParentRightsXML,
                                                     &keyRingDeviceStruct,
                                                     &TST_KeyDevice,
                                                     &keyRingKmacKrecParentStruct,
                                                     &TST_ParentKmacKey,
                                                     &TST_ParentKrecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);


    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall- Parent RO",
                               "TST_ATP_SEC_RO_InvalidParentROActivation",
                               "TST_ATP_SEC_RO_InvalidParentROActivation");
    for(index = 0;index < 2;index++ )
    {
        if (index == 0)
        {
            /*Initialize parent RO XML structure*/
            TST_ParentRightsXML.buffSizeInBytes = DX_VOS_StrLen(InvParentKeyRightsXmlStr);
            TST_ParentRightsXML.buff_ptr = (DxByte*)InvParentKeyRightsXmlStr;
        }
        else
        {
            /*Initialize parent RO XML structure*/
            TST_ParentRightsXML.buffSizeInBytes = DX_VOS_StrLen(InvParentIntegrityRightsXmlStr);
            TST_ParentRightsXML.buff_ptr = (DxByte*)InvParentIntegrityRightsXmlStr;
        }
        /*6-7. Activate content license using  TLK_ODRM_RetrieveKey() with invalid parent RO*/
        TST_Error = TLK_ODRM_RetrieveKey(&TST_ChildRightsXML,
                                         &keyRingKmacKrecStruct,
                                         &TST_kmacKey,
                                         &TST_krecKey,
                                         "cid:cid6121-2_jpg",
                                         &TST_ServiceDescr,
                                         &TST_ParentRightsXML,
                                         &keyRingKmacKrecParentStruct,
                                         &TST_ParentKmacKey,
                                         DX_NULL,
                                         0,
                                         TST_WrappedKey,
                                         RO_TST_gWorkspace,
                                         RO_TST_gWorkspaceSizeInBytes);

        /*8. Verify that TLK_ODRM_RC_ERROR_RO_MAC_VERIFICATION_FAILED error was returned*/
        if(TST_Error != TLK_ODRM_RC_ERROR_RO_MAC_VERIFICATION_FAILED)
            TST_Error = TST_FAIL;
        else
            TST_Error = DX_SUCCESS;

        TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_RetrieveKey with invalid parent RO succeed. Test ",
                                   "TST_ATP_SEC_RO_InvalidParentROActivation",
                                   "TST_ATP_SEC_RO_InvalidParentROActivation");

    }
EXIT_ON_ERROR:
    TLK_SCLK_Terminate(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    return TST_Error;
}



/*****************************************************************************
* Function Name:                                                           
*  TST_ATP_SEC_RO_InvalidROActivation
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return                                        
*
* TEST DESCRIPTION:
*  To test activation of license with invalid RO XML Secure RO Handling 
* Algorithm:
* 1. Get workspace size. 
* 2. Create key ring and initialize it with device private key
* 3. Call to  TLK_ODRM_ProcessRegistration() API  for RI Issuer registration
* 4. Call to TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key and RO XML.
* 5. Activate content license using  TLK_ODRM_RetrieveKey() with keys from step 4, secure clock descriptor from step 3and invalid RO XML (RO integrity).
* 6. Verify that ODRM_RC_ERROR_RO_MAC_VERIFICATION_FAILED error was returned
* 7.Activate content license using  TLK_ODRM_RetrieveKey() with keys from step 4, secure clock descriptor from step 3 and invalid key in RO XML.
* 8.Verify that TLK_ODRM_RC_ERROR_RO_MAC_VERIFICATION_FAILED error was returned
*******************************************************************************/   
DxError_t TST_ATP_SEC_RO_InvalidROActivation(void)
{

    DxUint32_t TST_Error = 0;
    DxUint8_t index = 0; 
    TLK_ODRM_KMNGKeyRing_t keyRingDeviceStruct = {0};

    TLK_ODRM_KMNGKeyRing_t keyRingKmacKrecStruct = {0};

    TLK_ODRM_KMNGKey_t  TST_KeyDevice = {0};
    TLK_ODRM_KMNGKey_t  TST_kmacKey = {0};
    TLK_ODRM_KMNGKey_t  TST_krecKey = {0};
    TLK_ODRM_Buffer_t   TST_RightsXML = {0};
    DxUint32_t  TST_KeyRingSize = 0;

    KMNG_AES_WrappedKey_t  TST_WrappedKey;

    KMNG_UserSpecificKeyData_t  TST_UserSpecificKeyData = {0,0};
    TLK_SCLK_ServiceClockDescriptor_t    TST_ServiceDescr = {0};

    DxUint32_t TST_NumOfSymKeys = 0;
    DxUint32_t TST_NumOfRSAKeys = 0;
    DxUint32_t TST_NumOfDHKeys  = 0;

    /*******************Set password to RI ID ****************/
    TLK_ODRM_Buffer_t passwordBuff;
    passwordBuff.buff_ptr        = TST_RiIDKey;
    passwordBuff.buffSizeInBytes = DX_VOS_StrLen(TST_RiIDKey);
    TST_KRING_Clean();

    ATP_LOG_TST_PRINT((MW_ATP_MSG,"\n******************************************** \n"));
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"***** TST_ATP_SEC_RO_InvalidROActivation ***\n"));     
    ATP_LOG_TST_PRINT((MW_ATP_MSG,"********************************************\n \n"));

    /*Initialize RO XML  struct*/
    TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(RightsXmlStr);
    TST_RightsXML.buff_ptr = (DxByte*)RightsXmlStr;

    /* 1. Get work space size */
    TST_Error = TLK_ODRM_WorkspaceSizeGet(TLK_ODRM_MIN_WORKSPACE, &RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_WorkspaceSizeGet",
                               "TST_ATP_SEC_RO_InvalidROActivation", 
                               "TST_ATP_SEC_RO_InvalidROActivation");


    if(RO_TST_gWorkspaceSizeInBytes < ODRMTLK_TST_WORKSPACE_SIZE)
        TST_Error = TST_FAIL;

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "Insufficient Workspace Size",
                               "TST_ATP_SEC_RO_InvalidROActivation",
                               "TST_ATP_SEC_RO_InvalidROActivation");


    /* 2. Create key ring and initialize it with device private key */
    TST_NumOfSymKeys = 0;
    TST_NumOfRSAKeys = 1;
    TST_NumOfDHKeys  = 0;

    TST_Error =  KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
                                  passwordBuff.buff_ptr,
                                  TST_NumOfSymKeys,
                                  TST_NumOfRSAKeys,
                                  TST_NumOfDHKeys,
                                  TST_gKeyRingDevice);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingDevice",
                               "TST_ATP_SEC_RO_InvalidROActivation",
                               "TST_ATP_SEC_RO_InvalidROActivation");

    TST_NumOfSymKeys = 2;
    TST_NumOfRSAKeys = 0;
    TST_NumOfDHKeys  = 0;

    TST_Error = KMNG_KeyRingInit(passwordBuff.buffSizeInBytes,
        passwordBuff.buff_ptr,
        TST_NumOfSymKeys,
        TST_NumOfRSAKeys,
        TST_NumOfDHKeys,
        TST_gKeyRingKmacKrec);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_KeyRingInit - TST_gKeyRingKmacKrec",
                               "TST_ATP_SEC_RO_InvalidROActivation",
                               "TST_ATP_SEC_RO_InvalidROActivation");

    TST_KeyDevice.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_KeyDevice.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_kmacKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_kmacKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    TST_krecKey.keyPassword_ptr =passwordBuff.buff_ptr;
    TST_krecKey.keyPasswordLenInBytes = passwordBuff.buffSizeInBytes;


    TST_Error = KMNG_ImportRsaUserKey(passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      passwordBuff.buff_ptr, 
                                      passwordBuff.buffSizeInBytes,
                                      KMNG_KeyTypeRSAPair,                    /*keyType*/
                                      ODRMTLK_TST_RSA_1024_KEY_SIZE_IN_BYTES, /*keySize*/ 
                                      KMNG_KEY_USAGE_ODRM | KMNG_KEY_USAGE_SIGNING,                    /*keyUsage*/
                                      TLK_ODRM_KEY_RESTRICTION,               /*keyRestriction*/
                                      TST_UserSpecificKeyData,                    /*UserSpecificKeyData*/
                                      PrivModulus,                           /*modulus_ptr*/
                                      MODULUS_SIZE,                       /*modulusSize*/
                                      PubExponent,                       /*E_ptr*/
                                      PUB_EXP_SIZE,                    /*E_Size*/
                                      PrivExponent,                       /*D_ptr*/
                                      PRIV_EXP_SIZE,                   /*D_Size*/
                                      &TST_KeyDevice.keyId,
                                      TST_gKeyRingDevice);


    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "KMNG_ImportRsaUserKey ",
                               "TST_ATP_SEC_RO_InvalidROActivation",
                               "TST_ATP_SEC_RO_InvalidROActivation");

    keyRingDeviceStruct.keyKmngRing_ptr = TST_gKeyRingDevice;
    keyRingDeviceStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingDeviceStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    keyRingKmacKrecStruct.keyKmngRing_ptr = TST_gKeyRingKmacKrec;
    keyRingKmacKrecStruct.keyKmngRingPassword_ptr = passwordBuff.buff_ptr;
    keyRingKmacKrecStruct.keyKmngRingPasswordLenInBytes = passwordBuff.buffSizeInBytes;

    /*3. Initialize secure cklock descriptor */
    TST_Error = TLK_SCLK_Init(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_SCLK_Init",
                               "TST_ATP_SEC_RO_InvalidROActivation",
                               "TST_ATP_SEC_RO_InvalidROActivation");

    TST_Error = TST_SCLK_Init(&TST_ServiceDescr);
    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TST_SCLK_Init",
                               "TST_ATP_SEC_RO_InvalidROActivation",
                               "TST_ATP_SEC_RO_InvalidROActivation");


    /*4. Call to  TLK_ODRM_VerifyDeviceRoBeforeInstall() API with the device private key */
    TST_Error = TLK_ODRM_VerifyDeviceRoBeforeInstall(&TST_RightsXML,
                                                     &keyRingDeviceStruct,
                                                     &TST_KeyDevice,
                                                     &keyRingKmacKrecStruct,
                                                     &TST_kmacKey,
                                                     &TST_krecKey,
                                                     RO_TST_gWorkspace,
                                                     RO_TST_gWorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_VerifyDeviceRoBeforeInstall",
                               "TST_ATP_SEC_RO_InvalidROActivation",
                               "TST_ATP_SEC_RO_InvalidROActivation");


    for (index = 0; index < 2; index++)
    {
        DxChar cidStr[16];

        if(index == 0)
        {
            /*Initialize rights  struct*/
            TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(InvIntegrityRightsXmlStr);
            TST_RightsXML.buff_ptr = (DxByte*)InvIntegrityRightsXmlStr;
            DX_VOS_StrNCopy(cidStr,16,"cid:cid6093_jpg");
        }
        else
        {
            /*Initialize rights  struct*/
            TST_RightsXML.buffSizeInBytes = DX_VOS_StrLen(InvalidKeyRightsXmlStr);
            TST_RightsXML.buff_ptr = (DxByte*)InvalidKeyRightsXmlStr;
            DX_VOS_StrNCopy(cidStr,16,"cid:cid6094_jpg");
        }

        /*5,7. Activate content license using  TLK_ODRM_RetrieveKey() with KMAC 
        and KREK keys from step 4 and secure clock descriptor from step 3.*/
        TST_Error = TLK_ODRM_RetrieveKey(&TST_RightsXML,
                                         &keyRingKmacKrecStruct,
                                         &TST_kmacKey,
                                         &TST_krecKey,
                                         cidStr,
                                         &TST_ServiceDescr,
                                         DX_NULL,
                                         DX_NULL,
                                         DX_NULL,
                                         DX_NULL,
                                         0,
                                         TST_WrappedKey,
                                         RO_TST_gWorkspace,
                                         RO_TST_gWorkspaceSizeInBytes);
       /*6,8. Verify that ODRM_RC_ERROR_RO_MAC_VERIFICATION_FAILED error was returned*/
        if(TST_Error != TLK_ODRM_RC_ERROR_RO_MAC_VERIFICATION_FAILED)
            TST_Error = TST_FAIL;
        else
            TST_Error = DX_SUCCESS;
        TST_ATP_CHECK_ERROR_RETURN(TST_Error, "TLK_ODRM_RetrieveKey succeed with invalid RO. Test ",
                                   "TST_ATP_SEC_RO_InvalidROActivation",
                                   "TST_ATP_SEC_RO_InvalidROActivation");
    }
EXIT_ON_ERROR:
    TLK_SCLK_Terminate(RO_TST_gWorkspace, RO_TST_gWorkspaceSizeInBytes);
    return TST_Error;
}
